#include <node.h>
#include <uv.h>
#include "face.h"
#include <functional>

namespace FaceDetection
{

    using namespace v8;
    using v8::Function;
    using v8::FunctionCallbackInfo;
    using v8::Isolate;
    using v8::HandleScope;
    using v8::Local;
    using v8::Null;
    using v8::Symbol;
    using v8::Object;
    using v8::String;
    using v8::Number;
    using v8::Int8Array;

    using v8::Value;
    using namespace std;

    Handle<Function> cb;
    uv_thread_t texample_thread;


    uv_async_t checkAsyncToken;
    uv_async_t saveAsyncToken;
    uv_async_t frameAsyncToken;

    uv_loop_t *loop = uv_default_loop();

    struct faceRequest
    {
        Persistent<Function> callback;
        Isolate *isolate;
        string filename;
        cv::Mat frame;
        int result;
    };

    FaceDetector *faceDetector = new FaceDetector();

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    ///
    /// \param args
    void Save(const FunctionCallbackInfo<Value> &args)
    {
        Isolate *isolate = args.GetIsolate();

        auto request = (faceRequest *) saveAsyncToken.data;
        auto cb = Local<Function>::Cast(args[1]);
        v8::String::Utf8Value filename(args[0]->ToString());

        request->isolate = isolate;
        request->callback.Reset(isolate, cb);

        faceDetector->SaveFace(*filename);
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    ///
    /// \param args
    void Check(const FunctionCallbackInfo<Value> &args)
    {
        Isolate *isolate = args.GetIsolate();

        auto request = (faceRequest *) checkAsyncToken.data;
        auto cb = Local<Function>::Cast(args[1]);
        v8::String::Utf8Value filename(args[0]->ToString());

        request->isolate = isolate;
        request->callback.Reset(isolate, cb);

        faceDetector->CheckFace(*filename);
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    ///
    /// \param args
    void Init(const FunctionCallbackInfo<Value> &args)
    {
        v8::String::Utf8Value param1(args[0]->ToString());
        std::string workingDir = std::string(*param1);

        saveAsyncToken.data = new faceRequest();
        checkAsyncToken.data = new faceRequest();

        faceDetector->Init(workingDir);

        uv_async_init(loop, &saveAsyncToken, [](uv_async_t *req)
        {
            auto request = (faceRequest *) req->data;
            auto isolate = request->isolate;
            HandleScope scope(isolate);

            printf("save callback dispatcher\n");
            auto fname = v8::String::NewFromUtf8(isolate, request->filename.c_str());

            const unsigned argc = 1;
            Local<Value> argv[argc] = {fname};
            Local<Function>::New(isolate, request->callback)->
                    Call(isolate->GetCurrentContext()->Global(), argc, argv);
        });

        uv_async_init(loop, &checkAsyncToken, [](uv_async_t *req)
        {
            auto request = (faceRequest *) req->data;
            auto isolate = request->isolate;
            HandleScope scope(isolate);

            auto fname = v8::String::NewFromUtf8(isolate, request->filename.c_str());
            auto votes = v8::Integer::New(isolate, request->result);

            const unsigned argc = 2;
            Local<Value> argv[argc] = {fname, votes};
            Local<Function>::New(isolate, request->callback)->
                    Call(isolate->GetCurrentContext()->Global(), argc, argv);
        });
    }

    ///
    /// \param args
    void Start(const FunctionCallbackInfo<Value> &args)
    {
        Isolate *isolate = args.GetIsolate();
        faceRequest *request = new faceRequest();
        cb = Local<Function>::Cast(args[0]);
        request->callback.Reset(isolate, cb);
        request->isolate = isolate;

        frameAsyncToken.data = request;
        uv_async_init(loop, &frameAsyncToken, [](uv_async_t *req)
        {
            auto request = (faceRequest *) req->data;
            auto isolate = request->isolate;
            auto frame = request->frame;
            HandleScope scope(isolate);

            size_t size = (size_t) frame.rows * 3;
            v8::Local<ArrayBuffer> buff = v8::ArrayBuffer::New(isolate, size * 2);
            v8::Local<Int16Array> array = v8::Int16Array::New(buff, 0, size);

            for (uint32_t i = 0; i < size; i += 3)
            {
                array->Set(i + 0, Integer::New(isolate, (int32_t) (frame.at<float>(i / 3, 0) * 1000)));
                array->Set(i + 1, Integer::New(isolate, (int32_t) (frame.at<float>(i / 3, 1) * 1000)));
                array->Set(i + 2, Integer::New(isolate, (int32_t) (frame.at<float>(i / 3, 2) * 1000)));
            }

            const unsigned argc = 1;
            Local<Value> argv[argc] = {array};
            Local<Function>::New(isolate, request->callback)->
                    Call(isolate->GetCurrentContext()->Global(), argc, argv);
        });
        uv_thread_create(&texample_thread, [](void *)
        {
            faceDetector->Start([](cv::Mat mat)
                                {
                                    ((faceRequest *) frameAsyncToken.data)->frame = mat;
                                    uv_async_send(&frameAsyncToken);
                                },
                                [](std::string filename) //save callback
                                {
                                    printf("save callback \n");
                                    ((faceRequest *) (saveAsyncToken.data))->filename = filename;
                                    uv_async_send(&saveAsyncToken);
                                },
                                [](std::string filename, int votes) //check callback
                                {
                                    printf("check callback \n");
                                    ((faceRequest *) (checkAsyncToken.data))->filename = filename;
                                    ((faceRequest *) (checkAsyncToken.data))->result = votes;
                                    uv_async_send(&checkAsyncToken);
                                });
        }, NULL);
    }

    void Stop(const FunctionCallbackInfo<Value> &args)
    {
        faceDetector->Stop();
    }

    void init(Local<Object> exports, Local<Object> module)
    {
        NODE_SET_METHOD(exports, "saveFace", Save);
        NODE_SET_METHOD(exports, "checkFace", Check);
        NODE_SET_METHOD(exports, "start", Start);
        NODE_SET_METHOD(exports, "stop", Stop);
        NODE_SET_METHOD(exports, "setWorkingDir", Init);
    }

    NODE_MODULE(addon, init)
}