var tilesets = {};
var tilesetUidHash = {};

var Levels = {};
var fieldGetters = {
	Point: function(field, level) {
		return {cell: field.__value, position: [field.__value[0]*level.gridSize, field.__value[1]*level.gridSize]};
	},
	String: function(field) {
		return field.__value;
	}
};
var getValue = function(field, layer) {
	var type = field.__type || 'String';
	if(!(type in fieldGetters)){
		console.warn('No getter for', type, field);
		type = 'String';
	}
	return fieldGetters[type](field, layer);
};
var imagesHash = {};
var imageHashDestinations = {};
var colorFromHex = function(hex) {
	hex = hex.replace(/^#/,'');
	if(hex.length === 3)
		hex = hex[0]+hex[0] + hex[1]+hex[1] + hex[2]+hex[2];
	return [parseInt(hex[0]+hex[1], 16),parseInt(hex[2]+hex[3], 16),parseInt(hex[4]+hex[5], 16)];
};

var getImage = function(path, obj) {
	var img;
	if(imagesHash[path]){
		imageHashDestinations[path].push(obj);
		img = imagesHash[path];
	}else{
		img = document.createElement( 'img' );
		img.onload = function() {
			var canvas = document.createElement('canvas');

			var w = canvas.width = img.naturalWidth,
					h = canvas.height = img.naturalHeight;

			var ctx = canvas.getContext('2d');
			ctx.drawImage(img, 0, 0);
			var data = ctx.getImageData(0,0, w, h );
			var color = colorFromHex(gameConfig.transparent || '#0f0'),
					dataEl = data.data;
			for( var i = 0, _i = dataEl.length; i < _i; i+=4 ){
				if(dataEl[ i ] === color[0] &&
					dataEl[ i+1 ] === color[1] &&
					dataEl[ i+2 ] === color[2])
					dataEl[i+3] = 0;
			}
			ctx.putImageData(data, 0,0);
			imagesHash[path] = canvas;
			for( var i = 0, _i = imageHashDestinations[path].length; i < _i; i++ ){
				var element = imageHashDestinations[path][ i ];
				element.img = canvas;
			}
		}
		imageHashDestinations[path] = [obj];
		img.src = path;
		imagesHash[path] = img;
	}
	return img;
}
var loadWorld = function(url){


	World.fromURL( url ).then( async world => {
		Levels = world.levelMap;


		world.tilesetIds.forEach( tilesetID => {
			var tileset = tilesets[ tilesetID ] = new Tileset( world, world.tilesetMap[ tilesetID ].data );
			tileset.img = getImage(tileset.path, tileset);
			tilesetUidHash[ tileset.uid ] = tileset;
		} );

		for( var levelName in Levels ){
			var level = Levels[ levelName ];

			level.layerHash = {};
			level.getItems = function(type) {
				return this.entitiesTypes[type] || [];
			}
			var entities = level.entities = [];
			var entitiesTypes = level.entitiesTypes = {};

			for( var i = 0, _i = level.layers.length; i < _i; i++ ){
				var layer = level.layers[ i ];
				var layerObj = {
					autoLayerTiles: layer.data.autoLayerTiles.concat(layer.data.gridTiles),
					gridSize: layer.gridSize,
					__identifier: layer.data.__identifier,
					name: layer.data.__identifier,
					entities: layer.entities,
					tileset: tilesetUidHash[ layer.data.__tilesetDefUid ],
					width: layer.data.__cWid,
					height: layer.data.__cHei,
					grid: layer.data.intGrid
				};
				layer = level.layers[ i ] = layerObj;
				level.layerHash[ layer.name ] = layerObj;


				if( layer.entities ){
					for( var j = 0, _j = layer.entities.length; j < _j; j++ ){
						var entity = layer.entities[ j ];
						var obj = {
							id: entity.data.defUid,
							type: entity.data.__identifier,
							tile: entity.data.__tile,
							width: entity.data.width,
							cellWidth: entity.data.width / layer.gridSize,
							height: entity.data.height,
							cellHeight: entity.data.height / layer.gridSize,
							position: entity.data.px,
							cell: entity.data.__grid,
							level: level
						};
						if( obj.tile && obj.tile.tilesetUid ){
							obj.tile.tileset = tilesetUidHash[ obj.tile.tilesetUid ];
						}
						entity.data.fieldInstances.forEach( field => {
							obj[ field.__identifier ] = getValue( field, layer );
						} );

						entities.push( obj );
					}
				}
				layer.entities = [];
			}

			for( var i = 0, _i = entities.length; i < _i; i++ ){
				var entity = entities[ i ];
				if( entity.Layer ){
					entity.layer = level.layerHash[ entity.Layer ];
					if( gameConfig.objects[ entity.type ] ){
						entities[ i ] = entity = new gameConfig.objects[ entity.type ]( entity );
					}
					level.layerHash[ entity.Layer ].entities.push( entity );
					( entitiesTypes[ entity.type ] || ( entitiesTypes[ entity.type ] = [] ) ).push( entity );

				}
			}
			for( var i = 0, _i = entities.length; i < _i; i++ ){
				var entity = entities[ i ];
				entity.init && entity.init();
			}

		}


	} );
};
var gameConfig = {objects: {}};
var Game = function(cfg) {
	gameConfig = cfg;
	loadWorld(cfg.file);
};
Game.debugs = [];
var t = +new Date(), t2, dt, tfs = 0;
function loop(time) {
	t2 = +new Date();
	dt = (t2 - t)/1000;
	t = t2;
	if(dt>0.05)
		dt = 0.05;

	tfs += dt;

	gameConfig.step && gameConfig.step(dt, tfs);
	gameConfig.draw && gameConfig.draw(dt, tfs);
	for( var i = 0, _i = Game.debugs.length; i < _i; i++ ){
		Game.debugs[ i ]();
	}
	Game.debugs = [];
	window.requestAnimationFrame(loop);
}
window.requestAnimationFrame(loop)

var mainCanvas = c1;
var ctx = c1.getContext('2d');
ctx.imageSmoothingEnabled = false;

var w = 1600;
var h = 1024;

var Camera = {
	position: [w/2, h/2],
	scale: 8,
	transformX: function(x) {
		return ((x-this.position[0]*this.scale+w/2)|0);
	},
	transformY: function(y) {
		return y-this.position[1]*this.scale+h/2 |0;
	}
};

var _renderTile = function(tile) {

};
var renderTile = function(layer, tile) {
	renderTiles(layer, [tile]);
}


var drawImageFlipsRotates = function(img, sx,sy,sw,sh, dx,dy,dw,dh, flip, rotate) {
	dx = Camera.transformX(dx);
	dy = Camera.transformY(dy);
	if(dx>w || dy > h || dx+dw<0 || dy+dh<0)
		return;
	dw|=0;
	dh|=0;
	dx|=0;
	dy|=0;
	if(flip === 1){
		ctx.save();
		ctx.translate( dx + dw, dy );
		ctx.scale( -1, 1 );
		ctx.drawImage( img, sx,sy,sw,sh, 0,0,dw,dh );
		ctx.restore();
	}else{
		ctx.drawImage( img, sx, sy, sw, sh, dx, dy, dw, dh );
	}
};

var renderTiles = function(layer, tiles) {
	var scale = Camera.scale;
	var tileset = layer.tileset,//tilesetUidHash[layer.__tilesetDefUid],
			gridSize = tileset && tileset.gridSize,
			scaledGridSize = gridSize*scale;

	if(tiles && tiles.length){
		for( var i = 0, _i = tiles.length; i < _i; i++ ){
			var tile = tiles[ i ];
			drawImageFlipsRotates(
				tileset.img,
				tile.src[ 0 ], tile.src[ 1 ], gridSize, gridSize,
				tile.px[ 0 ] * scale, tile.px[ 1 ] * scale, scaledGridSize, scaledGridSize,
				tile.f
			);
		}
	}
};

var renderObject = function(obj) {
	var scale = Camera.scale;
	var tileset = obj.tile.tileset,
			gridSize = tileset.gridSize,
			tile = obj.tile;

	drawImageFlipsRotates(
		tileset.img,
		tile.srcRect[ 0 ], tile.srcRect[ 1 ], tile.srcRect[ 2 ], tile.srcRect[ 3 ],
		obj.position[0] * scale, obj.position[1] * scale, obj.width*scale, obj.height*scale,
		tile.f
	)
};
var clear = function() {
	ctx.fillStyle = gameConfig.background || '#171c39';
	ctx.fillRect(0,0,w,h);
};
var render = function(level) {
	if(!level)
		return;


	var layers = level.layers;
	for( var l = layers.length; l--; ){
		var layer = layers[ l ];
		renderTiles(layer, layer.autoLayerTiles);

		var entities = layer.entities;
		if(entities){
			for( var i = 0, _i = entities.length; i < _i; i++ ){
				var entity = entities[ i ];
				entity.draw && entity.draw(ctx, layer);

			}
		}
	}
};

var physic = function(level, dt, tfs) {
	if(!level)
		return;

	var layers = level.layers;
	for( var l = layers.length; l--; ){
		var layer = layers[ l ];

		var t = +new Date();
		var entities = layer.entities;
		if(entities){
			for( var i = 0, _i = entities.length; i < _i; i++ ){
				var entity = entities[ i ];
				entity.step && entity.step(dt, tfs);
				if(entity.animations){
					for( var a = entity.animations.length - 1; a >= 0; a-- ){
						var animation = entity.animations[ a ],
							percent = Math.max( 0, Math.min( 1, ( t / 1000 - ( animation.start + animation.wait ) ) / animation.duration ) );
						animation.fn.call( entity, percent );
						if( percent === 1 ){
							entity.animations.splice( a, 1 );
						}
					}
				}
			}
		}
	}
};

var GameObject = function(cfg) {
	Object.assign(this, cfg);
	this.collisionOffset = [0,0,0,0];
	this.animations = [];
	this.animationsHash = {};
	this.spriteAnimations = {};
	this.saying = false;
};
GameObject.prototype = {
	visible: true,
	opacity: 1,
	say: function( sentence, duration) {
		duration = duration || sentence.length/8;
		var actions = [];
		sentence = sentence.split('\n');

		var maxLength = 0;
		for( var i = 0, _i = sentence.length; i < _i; i++ ){

			var sentenceElement = sentence[ i ];
			sentence[ i ] = sentenceElement = sentenceElement.replace(/\[>([^\]]+)\]/g, function(a,b){
				actions.push(b);
				return '';
			})
			maxLength = Math.max(sentenceElement.length, maxLength);
		}
		this.saying = {
			sentence: sentence.join('\n'),
			duration,
			maxLength,
			rowsCount: sentence.length,
			length: sentence.join('\n').length,
			start: +new Date(),
			currentChar: 0,
			actions
		};

	},
	sayRender: function(cfg) {
		if(this.saying){
			var saying = this.saying;
			var currentChar = (+new Date() - saying.start)/1000 / (saying.duration/saying.length);
			saying.currentChar = Math.floor(currentChar);
			saying.currentChar = Math.min(Math.max(0, saying.currentChar), saying.length+4);

			var part = saying.sentence.substr(0, saying.currentChar);

			var place = cfg.place || [-1, -1];
			var position = cfg.position || [0, 0];
			var padding = cfg.padding || [2, 2];
			var offset = cfg.offset || [0, 2];

			var symbolWidth = 2, symbolHeight = 5;
			if(place[0] === -1){
				var sx = ( this.position[ 0 ]+position[0] - ( saying.maxLength ) * symbolWidth ) * Camera.scale;
			}
			if(place[1] === -1){
				var sy = ( this.position[ 1 ]+position[1] - saying.rowsCount * symbolHeight ) * Camera.scale;
			}
			if(cfg.background){
				ctx.fillStyle = cfg.background;
				ctx.fillRect(
					Camera.transformX(sx-(padding[0]+offset[0])*Camera.scale),
					Camera.transformY(sy-(padding[1]+offset[1])*Camera.scale),
					saying.maxLength * symbolWidth * Camera.scale+padding[0]*2*Camera.scale,
					saying.rowsCount * symbolHeight * Camera.scale +padding[1]*2*Camera.scale)
			}
			if(cfg.border){
				ctx.strokeStyle = cfg.border;
				ctx.lineWidth = Camera.scale
				ctx.strokeRect(
					Camera.transformX(sx-(padding[0]+offset[0])*Camera.scale),
					Camera.transformY(sy-(padding[1]+offset[1])*Camera.scale),
					saying.maxLength * symbolWidth * Camera.scale+padding[0]*2*Camera.scale,
					saying.rowsCount * symbolHeight * Camera.scale +padding[1]*2*Camera.scale)

			}



			ctx.fillStyle = cfg.color || '#fff';
			ctx.font = '24px Tahoma';
			var currentRow = 0, currentCol = 0;
			for(var i = 0, _i = part.length; i< _i; i++){
				var entropy = _i/(currentChar-i)*8;
				if(part[i] === '\n'){
					currentRow++;
					currentCol = 0;
					continue
				}
				ctx.fillText(part[i],
					Camera.transformX(sx + currentCol*symbolWidth * Camera.scale),
					Camera.transformY(sy+currentRow*symbolHeight* Camera.scale+Math.sin(entropy)),
				)
				currentCol++;
			}

			if(this.saying.currentChar >= this.saying.length+4){
				if(saying.actions){
					if(this.actions){
						saying.actions.forEach(act=>{
							if(this.actions[act]){
								this.actions[act].call(this);
							}else{
								console.error('No action called', act, 'in object', this)
							}
						})
					}
				}
				this.saying = false;
			}
		}
	},
	animate: function(name, fn, duration, wait) {
		if(!(name in this.animationsHash)){
			this.animationsHash[name] = true;
			this.animations.push( { fn, duration, wait: wait || 0, start: +new Date()/1000 } );
		}
	},
	wouldCollide: function(position, layers) {
		for( var i = 0, _i = layers.length; i < _i; i++ ){
			let layer = layers[ i ];

			layer = this.level.layerHash[layer];
			if(!layer){
				console.error( 'Layer', layers[ i ], 'does not exist' )
				continue
			}

			var grid = layer.grid;
			let gs = layer.gridSize; 
			for(var x = position[0]+this.collisionOffset[0], _x = Math.ceil((position[0]+this.width-this.collisionOffset[2])/gs)*gs; x < _x; x+= gs){
				for(var y = position[1]+this.collisionOffset[1], _y = Math.ceil((position[1]+this.height-this.collisionOffset[3])/gs)*gs; y < _y; y+= gs){
					let cx = Math.floor(x/gs),
							cy = Math.floor(y/gs);
					if(this.DEBUG){
						Game.debugs.push( () => {
							var scale = Camera.scale;
							ctx.strokeStyle = '#e25002';

							ctx.strokeRect( Camera.transformX(cx * gs * scale), Camera.transformY(cy * gs * scale), gs * scale, gs * scale )
						} );
					}
					if(grid[cx+cy*layer.width])
						return true;

				}
			}
		}
		return false;
	},
	terraform: function(tiles, state) {
		if(this._terraformState !== state){
			this._terraformState = state;
			var gridSize = this.layer.gridSize
			for( var i = 0, _i = tiles.length; i < _i; i++ ){
				var tile = tiles[ i ];
				this.layer.grid[tile.px[0]/gridSize+tile.px[1]/gridSize*this.layer.width] = state-0;
			}
		}
	},
	extractTiles: function() {
		this._terraformState = true;

		var allTiles = this.layer.autoLayerTiles;
		var out = [];
		for( var i = allTiles.length-1; i>-1;i-- ){
			var tile = allTiles[ i ];
			if(
				tile.px[0]>=this.position[0] &&
				tile.px[1]>=this.position[1] &&
				tile.px[0]<=this.position[0]+this.width &&
				tile.px[1]<=this.position[1]+this.height
			){
				out.push(allTiles.splice(i, 1)[0])
			}
		}
		return out;
	},
	extendSprite: function() {
		var sw = this.tile.srcRect[2];
		var sh = this.tile.srcRect[3];
		if(sw/this.width>sh/this.height){
			this.tile.srcRect[3] = sw/this.width*this.height;
		}else{
			this.tile.srcRect[2] = sh/this.height*this.width;
		}
	},
	render: function() {
		if(this.visible && this.opacity>0){
			if(this.opacity<1){
				ctx.save();
				ctx.globalAlpha = this.opacity;
			}
			renderObject( this );
			if(this.opacity<1){
				ctx.restore();
			}
		}
	},
	renderTiles: function(tiles) {
		if(this.visible && this.opacity>0){
			if( this.opacity < 1 ){
				ctx.save();
				ctx.globalAlpha = this.opacity;
			}
			renderTiles(this.layer, this.tiles);
			if( this.opacity < 1 ){
				ctx.restore();
			}
		}
	},
	draw: function() {
		this.render();
	},
	debug: function() {
		var scale = Camera.scale;
		var obj = this;
		ctx.strokeStyle = '#007bff';
		ctx.strokeRect(Camera.transformX(obj.position[0] * scale), Camera.transformY(obj.position[1] * scale), obj.width*scale, obj.height*scale);
		ctx.strokeStyle = '#ff00c8';
		ctx.strokeRect(Camera.transformX((obj.position[0]+obj.collisionOffset[0]) * scale),
			Camera.transformY((obj.position[1]+obj.collisionOffset[1]) * scale),
			(obj.width-obj.collisionOffset[0]-obj.collisionOffset[2])*scale,
			(obj.height-obj.collisionOffset[1]-obj.collisionOffset[3])*scale);
	},
	collide: function(type) {
		var its = this.level.getItems(typeof type === 'string' ? type : type.name);
		for( var i = 0, _i = its.length; i < _i; i++ ){
			var it = its[ i ];
			if(this.position[0]+this.width-this.collisionOffset[2] < it.position[0]+it.collisionOffset[0] ||
			this.position[0]+this.collisionOffset[0] > it.position[0] + it.width-it.collisionOffset[2] ||
			this.position[1]+this.height -this.collisionOffset[3]< it.position[1]+it.collisionOffset[1] ||
			this.position[1]+this.collisionOffset[1] > it.position[1] + it.height-it.collisionOffset[3]){
				continue;
			}

			return true;
		}
		return false;
	},
	getAll: function(type, filter) {
		var its = this.level.getItems(typeof type === 'string' ? type : type.name);
		if(filter){
			return its.filter(function(obj) {
				for(var i in filter){
					if(obj[i] !== filter[i])
						return false;
				}
				return true;
			})
		}
	},
	getItem: function(type, filter) {
		var its = this.level.getItems(typeof type === 'string' ? type : type.name);
		if(filter){
			return its.filter(function(obj) {
				for(var i in filter){
					if(obj[i] !== filter[i])
						return false;
				}
				return true;
			})[0]
		}
	},
	loadAnimation: function(animationName, {manual, duration, path, size, frames, offset, padding}) {
		var animation = this.spriteAnimations[animationName] = {
			name: animationName,
			size,
			frames,
			offset,
			padding,
			duration,
			currentFrame: 0,
			time: 0,
			manual
		};
		animation.img = getImage(path, animation);
		if(!this.currentAnimation){
			this.currentAnimation = animation;
		}
	},
	setAnimation: function(name, frame) {
		if(name !== false){
			this.currentAnimation = this.spriteAnimations[ name ];
		}
		var animation = this.currentAnimation;
		if( frame !== void 0 )
			animation.currentFrame = frame;

		animation.currentFrame = animation.currentFrame % animation.frames;
		this.tile = {
			f: this.tile.f,
			srcRect: [
				animation.currentFrame*animation.size[0], 0,
				animation.size[0], animation.size[1]
			],
			tileset: {
				img: animation.img,
				gridSize: animation.size[0]
			}
		};

	},
	updateAnimation: function(dt) {
		var animation = this.currentAnimation;
		animation.time += dt;

		if(animation.time>animation.duration)
			animation.time -= animation.duration;

		var frame = animation.manual ? animation.currentFrame : Math.floor(animation.time/(animation.duration/animation.frames));
		if(frame !== animation.currentFrame){
			this.setAnimation(false, frame);
			this.onFrameChange && this.onFrameChange(animation.name, frame)
		}

	}
};



var Keys = keys = {};
var keyMapping = {
	'ArrowUp': 'Up',
	'Space': 'Up',
	'w': 'Up',
	'KeyW': 'Up',

	'ArrowDown': 'Down',
	's': 'Down',
	'KeyS': 'Down',

	'ArrowRight': 'Right',
	'd': 'Right',
	'KeyD': 'Right',

	'ArrowLeft': 'Left',
	'a': 'Left',
	'KeyA': 'Left',

};
document.addEventListener('keydown', function(e){

	keys[keyMapping[e.code]] = true;
});
document.addEventListener('keyup', function(e){
	console.log(e.code)
	keys[keyMapping[e.code]] = false;
});