"use strict"; // good practice - see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Strict_mode
/**
 * @author Eric Haines / http://erichaines.com/
 *
 * Created for the Udacity course "Interactive Rendering", http://bit.ly/ericity
 *
 * Tessellate a teaspoon into triangular patches.
 *
 * See http://www.sjbaker.org/wiki/index.php?title=The_History_of_The_Teaspoon for
 *	the history of the Teaspoon and teaspoon
 *
 * THREE.TeaspoonGeometry = function ( size, segments )
 *
 * defaults: size = 50, segments = 10
 *
 * size is a relative scale: I've scaled the Teaspoon to fit vertically between -1 and 1.
 *	Think of it as a "radius".
 * segments - number of line segments to subdivide each patch edge;
 *	1 is possible but gives degenerates, so two is the real minimum.
 *
 * segments 'n' determines the number of objects output.
 *  Total patches = 16*2*n*n
 *
 *  size_factor  # triangles
 *       1			32
 *       2         128
 *       3         288
 *       4         512
 *       5         800
 *       6        1152
 *
 *		10        3200
 *		20       12800
 *		30       28800
 *		40       51200
 */
/*global THREE */

THREE.TeaspoonGeometry = function ( size, segments ) {
	"use strict";

	// 26 * 4 * 4 Bezier spline patches, note +1 start
	// Data from ftp://ftp.funet.fi/pub/sci/graphics/packages/objects/teaset.tar.Z
	var TeaspoonPatches = [
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,
33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,
49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,
65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,
81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,96,
97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,
113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,
129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,
145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,
161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,
177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,
193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,
209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,
225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,
241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,256
	] ;

	var TeaspoonVertices = [
-0.000107143,0.205357,0.0,
0.0,0.196429,-0.0178571,
0.0,0.196429,-0.0178571,
0.000107143,0.205357,0.0,
-0.0535714,0.205357,0.0,
-0.0222714,0.178571,-0.0534286,
0.0222714,0.178571,-0.0534286,
0.0535714,0.205357,0.0,
-0.107143,0.0952429,-0.0178571,
-0.0446429,0.0952429,-0.0892857,
0.0446429,0.0952429,-0.0892857,
0.107143,0.0952429,-0.0178571,
-0.107143,0.0,-0.0178571,
-0.0446429,0.0,-0.0892857,
0.0446429,0.0,-0.0892857,
0.107143,0.0,-0.0178571,
0.000107143,0.205357,0.0,
0.000135714,0.207589,0.00446429,
0.000157143,0.216518,0.00446429,
0.000125,0.214286,0.0,
0.0535714,0.205357,0.0,
0.0613964,0.212054,0.0133571,
0.0714286,0.220982,0.015625,
0.0625,0.214286,0.0,
0.107143,0.0952429,-0.0178571,
0.122768,0.0952429,0.0,
0.142857,0.0952429,0.00446429,
0.125,0.0952429,-0.0178571,
0.107143,0.0,-0.0178571,
0.122768,0.0,0.0,
0.142857,0.0,0.00446429,
0.125,0.0,-0.0178571,
0.000125,0.214286,0.0,
0.0,0.205357,-0.0178571,
0.0,0.205357,-0.0178571,
-0.000125,0.214286,0.0,
0.0625,0.214286,0.0,
0.0267857,0.1875,-0.0625,
-0.0267857,0.1875,-0.0625,
-0.0625,0.214286,0.0,
0.125,0.0952429,-0.0178571,
0.0535714,0.0952429,-0.107143,
-0.0535714,0.0952429,-0.107143,
-0.125,0.0952429,-0.0178571,
0.125,0.0,-0.0178571,
0.0535714,0.0,-0.107143,
-0.0535714,0.0,-0.107143,
-0.125,0.0,-0.0178571,
-0.000125,0.214286,0.0,
-0.000157143,0.216518,0.00446429,
-0.000135714,0.207589,0.00446429,
-0.000107143,0.205357,0.0,
-0.0625,0.214286,0.0,
-0.0714286,0.220982,0.015625,
-0.0613964,0.212054,0.0133571,
-0.0535714,0.205357,0.0,
-0.125,0.0952429,-0.0178571,
-0.142857,0.0952429,0.00446429,
-0.122768,0.0952429,0.0,
-0.107143,0.0952429,-0.0178571,
-0.125,0.0,-0.0178571,
-0.142857,0.0,0.00446429,
-0.122768,0.0,0.0,
-0.107143,0.0,-0.0178571,
-0.107143,0.0,-0.0178571,
-0.0446429,0.0,-0.0892857,
0.0446429,0.0,-0.0892857,
0.107143,0.0,-0.0178571,
-0.107143,-0.142857,-0.0178571,
-0.0446429,-0.142857,-0.0892857,
0.0446429,-0.142857,-0.0892857,
0.107143,-0.142857,-0.0178571,
-0.0133929,-0.160714,0.0386893,
-0.00557857,-0.160714,0.0386893,
0.00557857,-0.160714,0.0386893,
0.0133929,-0.160714,0.0386893,
-0.0133929,-0.25,0.0535714,
-0.00557857,-0.25,0.0535714,
0.00557857,-0.25,0.0535714,
0.0133929,-0.25,0.0535714,
0.107143,0.0,-0.0178571,
0.122768,0.0,0.0,
0.142857,0.0,0.00446429,
0.125,0.0,-0.0178571,
0.107143,-0.142857,-0.0178571,
0.122768,-0.142857,0.0,
0.142857,-0.142857,0.00446429,
0.125,-0.142857,-0.0178571,
0.0133929,-0.160714,0.0386893,
0.0153464,-0.160714,0.0386893,
0.0178571,-0.160714,0.0314357,
0.015625,-0.160714,0.0297607,
0.0133929,-0.25,0.0535714,
0.0153464,-0.25,0.0535714,
0.0178571,-0.25,0.0463179,
0.015625,-0.25,0.0446429,
0.125,0.0,-0.0178571,
0.0535714,0.0,-0.107143,
-0.0535714,0.0,-0.107143,
-0.125,0.0,-0.0178571,
0.125,-0.142857,-0.0178571,
0.0535714,-0.142857,-0.107143,
-0.0535714,-0.142857,-0.107143,
-0.125,-0.142857,-0.0178571,
0.015625,-0.160714,0.0297607,
0.00669643,-0.160714,0.0230643,
-0.00781071,-0.160714,0.0208321,
-0.015625,-0.160714,0.0297607,
0.015625,-0.25,0.0446429,
0.00669643,-0.25,0.0379464,
-0.00781071,-0.25,0.0357143,
-0.015625,-0.25,0.0446429,
-0.125,0.0,-0.0178571,
-0.142857,0.0,0.00446429,
-0.122768,0.0,0.0,
-0.107143,0.0,-0.0178571,
-0.125,-0.142857,-0.0178571,
-0.142857,-0.142857,0.00446429,
-0.122768,-0.142857,0.0,
-0.107143,-0.142857,-0.0178571,
-0.015625,-0.160714,0.0297607,
-0.0175786,-0.160714,0.0319929,
-0.0153464,-0.160714,0.0386893,
-0.0133929,-0.160714,0.0386893,
-0.015625,-0.25,0.0446429,
-0.0175786,-0.25,0.046875,
-0.0153464,-0.25,0.0535714,
-0.0133929,-0.25,0.0535714,
-0.0133929,-0.25,0.0535714,
-0.00557857,-0.25,0.0535714,
0.00557857,-0.25,0.0535714,
0.0133929,-0.25,0.0535714,
-0.0133929,-0.46425,0.0892857,
-0.00557857,-0.46425,0.0892857,
0.00557857,-0.46425,0.0892857,
0.0133929,-0.46425,0.0892857,
-0.0446429,-0.678571,0.0535714,
-0.00892857,-0.678571,0.0625,
0.00892857,-0.678571,0.0625,
0.0446429,-0.678571,0.0535714,
-0.0446429,-0.857143,0.0357143,
-0.00892857,-0.857143,0.0446429,
0.00892857,-0.857143,0.0446429,
0.0446429,-0.857143,0.0357143,
0.0133929,-0.25,0.0535714,
0.0153464,-0.25,0.0535714,
0.0178571,-0.25,0.0463179,
0.015625,-0.25,0.0446429,
0.0133929,-0.46425,0.0892857,
0.0153464,-0.464286,0.0892857,
0.0178571,-0.46425,0.0820321,
0.015625,-0.46425,0.0803571,
0.0446429,-0.678571,0.0535714,
0.0535714,-0.678571,0.0513393,
0.0535714,-0.678571,0.0334821,
0.0446429,-0.678571,0.0357143,
0.0446429,-0.857143,0.0357143,
0.0535714,-0.857143,0.0334821,
0.0535714,-0.857143,0.015625,
0.0446429,-0.857143,0.0178571,
0.015625,-0.25,0.0446429,
0.00669643,-0.25,0.0379464,
-0.00781071,-0.25,0.0357143,
-0.015625,-0.25,0.0446429,
0.015625,-0.46425,0.0803571,
0.00669643,-0.464286,0.0736607,
-0.00781071,-0.46425,0.0714286,
-0.015625,-0.46425,0.0803571,
0.0446429,-0.678571,0.0357143,
0.00892857,-0.678571,0.0446429,
-0.00892857,-0.678571,0.0446429,
-0.0446429,-0.678571,0.0357143,
0.0446429,-0.857143,0.0178571,
0.00892857,-0.857143,0.0267857,
-0.00892857,-0.857143,0.0267857,
-0.0446429,-0.857143,0.0178571,
-0.015625,-0.25,0.0446429,
-0.0175786,-0.25,0.046875,
-0.0153464,-0.25,0.0535714,
-0.0133929,-0.25,0.0535714,
-0.015625,-0.46425,0.0803571,
-0.0175786,-0.464286,0.0825893,
-0.0153464,-0.464286,0.0892857,
-0.0133929,-0.46425,0.0892857,
-0.0446429,-0.678571,0.0357143,
-0.0535714,-0.678571,0.0334821,
-0.0535714,-0.678571,0.0513393,
-0.0446429,-0.678571,0.0535714,
-0.0446429,-0.857143,0.0178571,
-0.0535714,-0.857143,0.015625,
-0.0535714,-0.857143,0.0334821,
-0.0446429,-0.857143,0.0357143,
-0.0446429,-0.857143,0.0357143,
-0.00892857,-0.857143,0.0446429,
0.00892857,-0.857143,0.0446429,
0.0446429,-0.857143,0.0357143,
-0.0446429,-0.928571,0.0285714,
-0.00892857,-0.928571,0.0375,
0.00892857,-0.928571,0.0375,
0.0446429,-0.928571,0.0285714,
-0.0539286,-0.999643,0.0178571,
0.000357143,-0.999643,0.0178571,
0.0,-0.999643,0.0178571,
0.0535714,-0.999643,0.0178571,
-0.000357143,-1,0.0178571,
0.000357143,-1,0.0178571,
0.0,-1,0.0178571,
0.0,-1,0.0178571,
0.0446429,-0.857143,0.0357143,
0.0535714,-0.857143,0.0334821,
0.0535714,-0.857143,0.015625,
0.0446429,-0.857143,0.0178571,
0.0446429,-0.928571,0.0285714,
0.0535714,-0.928571,0.0263393,
0.0535714,-0.928571,0.00848214,
0.0446429,-0.928571,0.0107143,
0.0535714,-0.999643,0.0178571,
0.0669643,-0.999643,0.0178571,
0.0673214,-0.999643,0.0,
0.0539286,-0.999643,0.0,
0.0,-1,0.0178571,
0.0,-1,0.0178571,
0.000357143,-1,0.0,
0.000357143,-1,0.0,
0.0446429,-0.857143,0.0178571,
0.00892857,-0.857143,0.0267857,
-0.00892857,-0.857143,0.0267857,
-0.0446429,-0.857143,0.0178571,
0.0446429,-0.928571,0.0107143,
0.00892857,-0.928571,0.0196429,
-0.00892857,-0.928571,0.0196429,
-0.0446429,-0.928571,0.0107143,
0.0539286,-0.999643,0.0,
0.000357143,-0.999643,0.0,
-0.000357143,-0.999643,0.0,
-0.0539286,-0.999643,0.0,
0.000357143,-1,0.0,
0.000357143,-1,0.0,
-0.000357143,-1,0.0,
-0.000357143,-1,0.0,
-0.0446429,-0.857143,0.0178571,
-0.0535714,-0.857143,0.015625,
-0.0535714,-0.857143,0.0334821,
-0.0446429,-0.857143,0.0357143,
-0.0446429,-0.928571,0.0107143,
-0.0535714,-0.928571,0.00848214,
-0.0535714,-0.928571,0.0263393,
-0.0446429,-0.928571,0.0285714,
-0.0539286,-0.999643,0.0,
-0.0673214,-0.999643,0.0,
-0.0675,-0.999643,0.0178571,
-0.0539286,-0.999643,0.0178571,
-0.000357143,-1,0.0,
-0.000357143,-1,0.0,
-0.000535714,-1,0.0178571,
-0.000357143,-1,0.0178571
	] ;

	var minPatches = 0;
	var maxPatches = 16;

	THREE.Geometry.call( this );

	this.size = size || 50;

	// number of segments per patch
	this.segments = Math.max( 2, Math.floor( segments ) || 10 );

	// scale the size to be the real scaling factor
	var maxHeight = 0.21463862761855126;
	var minHeight = -1;

	var fullHeight = maxHeight - minHeight;
	var fullHeight2 = fullHeight/2;
	var heightOffset = - fullHeight2 - minHeight;
	var trueSize = this.size / fullHeight2;

	var normals = [], uvs = [];
	// bezier form
	var ms = new THREE.Matrix4(  -1.0,  3.0, -3.0,  1.0,
                                  3.0, -6.0,  3.0,  0.0,
                                 -3.0,  3.0,  0.0,  0.0,
                                  1.0,  0.0,  0.0,  0.0 ) ;

	var g = [];
	var i, r, c;

	var sp = [];
	var tp = [];
	var dsp = [];
	var dtp = [];

	// M * G * M matrix, sort of see
	// http://www.cs.helsinki.fi/group/goa/mallinnus/curves/surfaces.html
	var mgm = [];

	var vert = [];
	var sdir = [];
	var tdir = [];

	var norm = new THREE.Vector3();

	var tcoord;

	var sstep, tstep;
	var gmx, tmtx;

	var vertPerRow;

	var s, t, sval, tval, p, dsval, dtval;

	var vsp, vtp, vdsp, vdtp;
	var vsdir, vtdir, vertOut;
	var v1, v2, v3, v4;

	var mst = ms.clone();
	mst.transpose();

	// internal function: test if triangle has any matching vertices;
	// if so, don't output, since it won't display anything.
	var notDegenerate = function ( vtx1, vtx2, vtx3 ) {
		if ( vtx1.equals( vtx2 ) ) { return false; }
		if ( vtx1.equals( vtx3 ) ) { return false; }
		if ( vtx2.equals( vtx3 ) ) { return false; }
		return true;
	};


	for ( i = 0; i < 3; i++ )
	{
		mgm[i] = new THREE.Matrix4();
	}

	vertPerRow = (this.segments+1);

	var surfCount = 0;
	//var faceCount = 0;

	for ( var surf = minPatches ; surf < maxPatches ; surf++ ) {
		// get M * G * M matrix for x,y,z
		for ( i = 0 ; i < 3 ; i++ ) {
			// get control patches
			for ( r = 0 ; r < 4 ; r++ ) {
				for ( c = 0 ; c < 4 ; c++ ) {
					// transposed; note subtraction of 1 for index
					g[c*4+r] = TeaspoonVertices[(TeaspoonPatches[surf*16 + r*4 + c]-1)*3 + i] ;
				}
			}

			// Shockingly, the following three.js does NOT work. Setting this way appears to give the order
			// g[0], g[4], g[8], etc. to the elements! I could avoid the transpose above
			// and things would "just work", but this weird ordering would be mysterious.
			//var gmx = new THREE.Matrix4();
			//gmx.elements.set( g );
			// So, explicitly set the matrix this way:
			gmx = new THREE.Matrix4( g[0], g[1], g[2], g[3], g[4], g[5], g[6], g[7], g[8], g[9], g[10], g[11], g[12], g[13], g[14], g[15] );

			tmtx = new THREE.Matrix4();
			tmtx.multiplyMatrices( gmx, ms );
			mgm[i].multiplyMatrices( mst, tmtx );
		}

		// step along, get points, and output
		for ( sstep = 0 ; sstep <= this.segments ; sstep++ ) {
			s = sstep / this.segments;

			for ( tstep = 0 ; tstep <= this.segments ; tstep++ ) {
				t = tstep / this.segments;

				// point from basis
				// get power vectors and their derivatives
				for ( p = 4, sval = tval = 1.0 ; p-- ; ) {
					sp[p] = sval ;
					tp[p] = tval ;
					sval *= s ;
					tval *= t ;

					if ( p === 3 ) {
						dsp[p] = dtp[p] = 0.0 ;
						dsval = dtval = 1.0 ;
					} else {
						dsp[p] = dsval * (3-p) ;
						dtp[p] = dtval * (3-p) ;
						dsval *= s ;
						dtval *= t ;
					}
				}

				vsp = new THREE.Vector4( sp[0], sp[1], sp[2], sp[3] );
				vtp = new THREE.Vector4( tp[0], tp[1], tp[2], tp[3] );
				vdsp = new THREE.Vector4( dsp[0], dsp[1], dsp[2], dsp[3] );
				vdtp = new THREE.Vector4( dtp[0], dtp[1], dtp[2], dtp[3] );

				// do for x,y,z
				for ( i = 0 ; i < 3 ; i++ ) {
					// multiply power vectors times matrix to get value
					tcoord = vsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					vert[i] = tcoord.dot( vtp );

					// get s and t tangent vectors
					tcoord = vdsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					sdir[i] = tcoord.dot( vtp ) ;

					tcoord = vsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					tdir[i] = tcoord.dot( vdtp ) ;
				}

				// find normal
				vsdir = new THREE.Vector3( sdir[0], sdir[1], sdir[2] );
				vtdir = new THREE.Vector3( tdir[0], tdir[1], tdir[2] );
				norm.crossVectors( vtdir, vsdir );
				norm.normalize();

				// rotate on X axis
				// interestingly, normals need to be reversed; I suspect the patch
				// has opposite handedness from the teapot. Also, Y is along axis for the teaspoon.
				normals.push( new THREE.Vector3( -norm.x, -norm.y, -norm.z ) );

				// TODO: check texturing
				uvs.push( new THREE.Vector2( 1-t, 1-s ) );

				// three.js uses Y up, the code makes Y up, all is fine.
				// Move teaspoon to be centered around origin, three.js style.
				vertOut = new THREE.Vector3( trueSize*vert[0], trueSize*(vert[1]+heightOffset), trueSize*vert[2] );

				this.vertices.push( vertOut );

			}
		}

		// save the faces
		for ( sstep = 0 ; sstep < this.segments ; sstep++ ) {
			for ( tstep = 0 ; tstep < this.segments ; tstep++ ) {
				v1 = surfCount * vertPerRow * vertPerRow + sstep * vertPerRow + tstep;
				v2 = v1 + 1;
				v3 = v2 + vertPerRow;
				v4 = v1 + vertPerRow;

				if ( notDegenerate ( this.vertices[v1], this.vertices[v2], this.vertices[v3] ) ) {
					this.faces.push( new THREE.Face3( v1, v2, v3, [ normals[v1], normals[v2], normals[v3] ] ) );
					this.faceVertexUvs[ 0 ].push( [ uvs[v1], uvs[v2], uvs[v3] ] );
				}
				if ( notDegenerate ( this.vertices[v1], this.vertices[v3], this.vertices[v4] ) ) {
					this.faces.push( new THREE.Face3( v1, v3, v4, [ normals[v1], normals[v3], normals[v4] ] ) );
					this.faceVertexUvs[ 0 ].push( [ uvs[v1], uvs[v3], uvs[v4] ] );
				}
				//faceCount+=2;
			}
		}
		// increment only if a surface was used
		surfCount++;
	}

	this.computeCentroids();
	this.computeFaceNormals();
	this.mergeVertices();

};


THREE.TeaspoonGeometry.prototype = Object.create( THREE.Geometry.prototype );
