"use strict"; // good practice - see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Strict_mode
/**
 * @author Eric Haines / http://erichaines.com/
 *
 * Created for the Udacity course "Interactive Rendering", http://bit.ly/ericity
 *
 * Tessellate a teacup into triangular patches.
 *
 * See http://www.sjbaker.org/wiki/index.php?title=The_History_of_The_Teacup for
 *	the history of the Teacup and teacup
 *
 * THREE.TeacupGeometry = function ( size, segments )
 *
 * defaults: size = 50, segments = 10
 *
 * size is a relative scale: I've scaled the Teacup to fit vertically between -1 and 1.
 *	Think of it as a "radius".
 * segments - number of line segments to subdivide each patch edge;
 *	1 is possible but gives degenerates, so two is the real minimum.
 *
 * segments 'n' determines the number of objects output.
 *  Total patches = 26*2*n*n
 *
 *  size_factor  # triangles
 *       1			52
 *       2         208
 *       3         468
 *       4         832
 *       5        1300
 *       6        1872
 *
 *		10        5200
 *		20       20800
 *		30       46800
 *		40       83200
 */
/*global THREE */

THREE.TeacupGeometry = function ( size, segments ) {
	"use strict";

	// 26 * 4 * 4 Bezier spline patches, note +1 start
	// Data from ftp://ftp.funet.fi/pub/sci/graphics/packages/objects/teaset.tar.Z
	var TeacupPatches = [
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
4,17,18,19,8,20,21,22,12,23,24,25,16,26,27,28,
19,29,30,31,22,32,33,34,25,35,36,37,28,38,39,40,
31,41,42,1,34,43,44,5,37,45,46,9,40,47,48,13,
13,14,15,16,49,50,51,52,53,54,55,56,57,58,59,60,
16,26,27,28,52,61,62,63,56,64,65,66,60,67,68,69,
28,38,39,40,63,70,71,72,66,73,74,75,69,76,77,78,
40,47,48,13,72,79,80,49,75,81,82,53,78,83,84,57,
193,194,195,196,197,198,199,200,201,202,203,204,1,2,3,4,
196,205,206,207,200,208,209,210,204,211,212,213,4,17,18,19,
207,214,215,216,210,217,218,219,213,220,221,222,19,29,30,31,
216,223,224,193,219,225,226,197,222,227,228,201,31,41,42,1,
229,230,231,28,232,233,234,235,236,237,238,239,240,241,242,243,
28,244,245,229,235,246,247,232,239,248,249,236,243,250,251,240,
57,58,59,60,85,86,87,88,89,90,91,92,93,94,95,96,
60,67,68,69,88,97,98,99,92,100,101,102,96,103,104,105,
69,76,77,78,99,106,107,108,102,109,110,111,105,112,113,114,
78,83,84,57,108,115,116,85,111,117,118,89,114,119,120,93,
93,94,95,96,121,122,123,124,125,126,127,128,129,130,131,132,
96,103,104,105,124,133,134,135,128,136,137,138,132,139,140,141,
105,112,113,114,135,142,143,144,138,145,146,147,141,148,149,150,
114,119,120,93,144,151,152,121,147,153,154,125,150,155,156,129,
129,130,131,132,157,158,159,160,161,162,163,164,165,166,167,168,
132,139,140,141,160,169,170,171,164,172,173,174,168,175,176,177,
141,148,149,150,171,178,179,180,174,181,182,183,177,184,185,186,
150,155,156,129,180,187,188,157,183,189,190,161,186,191,192,165
	] ;

	var TeacupVertices = [
0.409091,0.772727,0.0,
0.409091,0.772727,-0.229091,
0.229091,0.772727,-0.409091,
0.0,0.772727,-0.409091,
0.409091,0.886364,0.0,
0.409091,0.886364,-0.229091,
0.229091,0.886364,-0.409091,
0.0,0.886364,-0.409091,
0.454545,0.886364,0.0,
0.454545,0.886364,-0.254545,
0.254545,0.886364,-0.454545,
0.0,0.886364,-0.454545,
0.454545,0.772727,0.0,
0.454545,0.772727,-0.254545,
0.254545,0.772727,-0.454545,
0.0,0.772727,-0.454545,
-0.229091,0.772727,-0.409091,
-0.409091,0.772727,-0.229091,
-0.409091,0.772727,0.0,
-0.229091,0.886364,-0.409091,
-0.409091,0.886364,-0.229091,
-0.409091,0.886364,0.0,
-0.254545,0.886364,-0.454545,
-0.454545,0.886364,-0.254545,
-0.454545,0.886364,0.0,
-0.254545,0.772727,-0.454545,
-0.454545,0.772727,-0.254545,
-0.454545,0.772727,0.0,
-0.409091,0.772727,0.229091,
-0.229091,0.772727,0.409091,
0.0,0.772727,0.409091,
-0.409091,0.886364,0.229091,
-0.229091,0.886364,0.409091,
0.0,0.886364,0.409091,
-0.454545,0.886364,0.254545,
-0.254545,0.886364,0.454545,
0.0,0.886364,0.454545,
-0.454545,0.772727,0.254545,
-0.254545,0.772727,0.454545,
0.0,0.772727,0.454545,
0.229091,0.772727,0.409091,
0.409091,0.772727,0.229091,
0.229091,0.886364,0.409091,
0.409091,0.886364,0.229091,
0.254545,0.886364,0.454545,
0.454545,0.886364,0.254545,
0.254545,0.772727,0.454545,
0.454545,0.772727,0.254545,
0.454545,0.545455,0.0,
0.454545,0.545455,-0.254545,
0.254545,0.545455,-0.454545,
0.0,0.545455,-0.454545,
0.454545,0.272727,0.0,
0.454545,0.272727,-0.254545,
0.254545,0.272727,-0.454545,
0.0,0.272727,-0.454545,
0.318182,0.0454545,0.0,
0.318182,0.0454545,-0.178182,
0.178182,0.0454545,-0.318182,
0.0,0.0454545,-0.318182,
-0.254545,0.545455,-0.454545,
-0.454545,0.545455,-0.254545,
-0.454545,0.545455,0.0,
-0.254545,0.272727,-0.454545,
-0.454545,0.272727,-0.254545,
-0.454545,0.272727,0.0,
-0.178182,0.0454545,-0.318182,
-0.318182,0.0454545,-0.178182,
-0.318182,0.0454545,0.0,
-0.454545,0.545455,0.254545,
-0.254545,0.545455,0.454545,
0.0,0.545455,0.454545,
-0.454545,0.272727,0.254545,
-0.254545,0.272727,0.454545,
0.0,0.272727,0.454545,
-0.318182,0.0454545,0.178182,
-0.178182,0.0454545,0.318182,
0.0,0.0454545,0.318182,
0.254545,0.545455,0.454545,
0.454545,0.545455,0.254545,
0.254545,0.272727,0.454545,
0.454545,0.272727,0.254545,
0.178182,0.0454545,0.318182,
0.318182,0.0454545,0.178182,
0.545455,0.0454545,0.0,
0.545455,0.0454545,-0.305455,
0.305455,0.0454545,-0.545455,
0.0,0.0454545,-0.545455,
0.727273,0.136364,0.0,
0.727273,0.136364,-0.407273,
0.407273,0.136364,-0.727273,
0.0,0.136364,-0.727273,
0.909091,0.136364,0.0,
0.909091,0.136364,-0.509091,
0.509091,0.136364,-0.909091,
0.0,0.136364,-0.909091,
-0.305455,0.0454545,-0.545455,
-0.545455,0.0454545,-0.305455,
-0.545455,0.0454545,0.0,
-0.407273,0.136364,-0.727273,
-0.727273,0.136364,-0.407273,
-0.727273,0.136364,0.0,
-0.509091,0.136364,-0.909091,
-0.909091,0.136364,-0.509091,
-0.909091,0.136364,0.0,
-0.545455,0.0454545,0.305455,
-0.305455,0.0454545,0.545455,
0.0,0.0454545,0.545455,
-0.727273,0.136364,0.407273,
-0.407273,0.136364,0.727273,
0.0,0.136364,0.727273,
-0.909091,0.136364,0.509091,
-0.509091,0.136364,0.909091,
0.0,0.136364,0.909091,
0.305455,0.0454545,0.545455,
0.545455,0.0454545,0.305455,
0.407273,0.136364,0.727273,
0.727273,0.136364,0.407273,
0.509091,0.136364,0.909091,
0.909091,0.136364,0.509091,
1.0,0.136364,0.0,
1.0,0.136364,-0.56,
0.56,0.136364,-1.0,
0.0,0.136364,-1.0,
1.0,0.0909091,0.0,
1.0,0.0909091,-0.56,
0.56,0.0909091,-1.0,
0.0,0.0909091,-1.0,
0.909091,0.0909091,0.0,
0.909091,0.0909091,-0.509091,
0.509091,0.0909091,-0.909091,
0.0,0.0909091,-0.909091,
-0.56,0.136364,-1.0,
-1.0,0.136364,-0.56,
-1.0,0.136364,0.0,
-0.56,0.0909091,-1.0,
-1.0,0.0909091,-0.56,
-1.0,0.0909091,0.0,
-0.509091,0.0909091,-0.909091,
-0.909091,0.0909091,-0.509091,
-0.909091,0.0909091,0.0,
-1.0,0.136364,0.56,
-0.56,0.136364,1.0,
0.0,0.136364,1.0,
-1.0,0.0909091,0.56,
-0.56,0.0909091,1.0,
0.0,0.0909091,1.0,
-0.909091,0.0909091,0.509091,
-0.509091,0.0909091,0.909091,
0.0,0.0909091,0.909091,
0.56,0.136364,1.0,
1.0,0.136364,0.56,
0.56,0.0909091,1.0,
1.0,0.0909091,0.56,
0.509091,0.0909091,0.909091,
0.909091,0.0909091,0.509091,
0.727273,0.0909091,0.0,
0.727273,0.0909091,-0.407273,
0.407273,0.0909091,-0.727273,
0.0,0.0909091,-0.727273,
0.545455,0.0,0.0,
0.545455,0.0,-0.305455,
0.305455,0.0,-0.545455,
0.0,0.0,-0.545455,
0.318182,0.0,0.0,
0.318182,0.0,-0.178182,
0.178182,0.0,-0.318182,
0.0,0.0,-0.318182,
-0.407273,0.0909091,-0.727273,
-0.727273,0.0909091,-0.407273,
-0.727273,0.0909091,0.0,
-0.305455,0.0,-0.545455,
-0.545455,0.0,-0.305455,
-0.545455,0.0,0.0,
-0.178182,0.0,-0.318182,
-0.318182,0.0,-0.178182,
-0.318182,0.0,0.0,
-0.727273,0.0909091,0.407273,
-0.407273,0.0909091,0.727273,
0.0,0.0909091,0.727273,
-0.545455,0.0,0.305455,
-0.305455,0.0,0.545455,
0.0,0.0,0.545455,
-0.318182,0.0,0.178182,
-0.178182,0.0,0.318182,
0.0,0.0,0.318182,
0.407273,0.0909091,0.727273,
0.727273,0.0909091,0.407273,
0.305455,0.0,0.545455,
0.545455,0.0,0.305455,
0.178182,0.0,0.318182,
0.318182,0.0,0.178182,
0.272727,0.0454545,0.0,
0.272727,0.0454545,-0.152727,
0.152727,0.0454545,-0.272727,
0.0,0.0454545,-0.272727,
0.409091,0.272727,0.0,
0.409091,0.272727,-0.229091,
0.229091,0.272727,-0.409091,
0.0,0.272727,-0.409091,
0.409091,0.545455,0.0,
0.409091,0.545455,-0.229091,
0.229091,0.545455,-0.409091,
0.0,0.545455,-0.409091,
-0.152727,0.0454545,-0.272727,
-0.272727,0.0454545,-0.152727,
-0.272727,0.0454545,0.0,
-0.229091,0.272727,-0.409091,
-0.409091,0.272727,-0.229091,
-0.409091,0.272727,0.0,
-0.229091,0.545455,-0.409091,
-0.409091,0.545455,-0.229091,
-0.409091,0.545455,0.0,
-0.272727,0.0454545,0.152727,
-0.152727,0.0454545,0.272727,
0.0,0.0454545,0.272727,
-0.409091,0.272727,0.229091,
-0.229091,0.272727,0.409091,
0.0,0.272727,0.409091,
-0.409091,0.545455,0.229091,
-0.229091,0.545455,0.409091,
0.0,0.545455,0.409091,
0.152727,0.0454545,0.272727,
0.272727,0.0454545,0.152727,
0.229091,0.272727,0.409091,
0.409091,0.272727,0.229091,
0.229091,0.545455,0.409091,
0.409091,0.545455,0.229091,
-0.454545,0.704545,0.0,
-0.454545,0.704545,-0.0454545,
-0.454545,0.772727,-0.0454545,
-0.772727,0.863636,0.0,
-0.772727,0.863636,-0.0454545,
-0.818182,0.954545,-0.0454545,
-0.818182,0.954545,0.0,
-0.772727,0.522727,0.0,
-0.772727,0.522727,-0.0454545,
-0.909091,0.477273,-0.0454545,
-0.909091,0.477273,0.0,
-0.409091,0.363636,0.0,
-0.409091,0.363636,-0.0454545,
-0.409091,0.295455,-0.0454545,
-0.409091,0.295455,0.0,
-0.454545,0.772727,0.0454545,
-0.454545,0.704545,0.0454545,
-0.818182,0.954545,0.0454545,
-0.772727,0.863636,0.0454545,
-0.909091,0.477273,0.0454545,
-0.772727,0.522727,0.0454545,
-0.409091,0.295455,0.0454545,
-0.409091,0.363636,0.0454545
	] ;

	var minPatches = 0;
	var maxPatches = 26;

	THREE.Geometry.call( this );

	this.size = size || 50;

	// number of segments per patch
	this.segments = Math.max( 2, Math.floor( segments ) || 10 );

	// scale the size to be the real scaling factor
	var maxHeight = 0.857954740524292;

	var maxHeight2 = maxHeight/2;
	var trueSize = this.size / maxHeight2;

	var normals = [], uvs = [];
	// bezier form
	var ms = new THREE.Matrix4(  -1.0,  3.0, -3.0,  1.0,
                                  3.0, -6.0,  3.0,  0.0,
                                 -3.0,  3.0,  0.0,  0.0,
                                  1.0,  0.0,  0.0,  0.0 ) ;

	var g = [];
	var i, r, c;

	var sp = [];
	var tp = [];
	var dsp = [];
	var dtp = [];

	// M * G * M matrix, sort of see
	// http://www.cs.helsinki.fi/group/goa/mallinnus/curves/surfaces.html
	var mgm = [];

	var vert = [];
	var sdir = [];
	var tdir = [];

	var norm = new THREE.Vector3();

	var tcoord;

	var sstep, tstep;
	var gmx, tmtx;

	var vertPerRow;

	var s, t, sval, tval, p, dsval, dtval;

	var vsp, vtp, vdsp, vdtp;
	var vsdir, vtdir, vertOut;
	var v1, v2, v3, v4;

	var mst = ms.clone();
	mst.transpose();

	// internal function: test if triangle has any matching vertices;
	// if so, don't output, since it won't display anything.
	var notDegenerate = function ( vtx1, vtx2, vtx3 ) {
		if ( vtx1.equals( vtx2 ) ) { return false; }
		if ( vtx1.equals( vtx3 ) ) { return false; }
		if ( vtx2.equals( vtx3 ) ) { return false; }
		return true;
	};


	for ( i = 0; i < 3; i++ )
	{
		mgm[i] = new THREE.Matrix4();
	}

	vertPerRow = (this.segments+1);

	var surfCount = 0;
	//var faceCount = 0;

	for ( var surf = minPatches ; surf < maxPatches ; surf++ ) {
		// get M * G * M matrix for x,y,z
		for ( i = 0 ; i < 3 ; i++ ) {
			// get control patches
			for ( r = 0 ; r < 4 ; r++ ) {
				for ( c = 0 ; c < 4 ; c++ ) {
					// transposed; note subtraction of 1 for index
					g[c*4+r] = TeacupVertices[(TeacupPatches[surf*16 + r*4 + c]-1)*3 + i] ;
				}
			}

			// Shockingly, the following three.js does NOT work. Setting this way appears to give the order
			// g[0], g[4], g[8], etc. to the elements! I could avoid the transpose above
			// and things would "just work", but this weird ordering would be mysterious.
			//var gmx = new THREE.Matrix4();
			//gmx.elements.set( g );
			// So, explicitly set the matrix this way:
			gmx = new THREE.Matrix4( g[0], g[1], g[2], g[3], g[4], g[5], g[6], g[7], g[8], g[9], g[10], g[11], g[12], g[13], g[14], g[15] );

			tmtx = new THREE.Matrix4();
			tmtx.multiplyMatrices( gmx, ms );
			mgm[i].multiplyMatrices( mst, tmtx );
		}

		// step along, get points, and output
		for ( sstep = 0 ; sstep <= this.segments ; sstep++ ) {
			s = sstep / this.segments;

			for ( tstep = 0 ; tstep <= this.segments ; tstep++ ) {
				t = tstep / this.segments;

				// point from basis
				// get power vectors and their derivatives
				for ( p = 4, sval = tval = 1.0 ; p-- ; ) {
					sp[p] = sval ;
					tp[p] = tval ;
					sval *= s ;
					tval *= t ;

					if ( p === 3 ) {
						dsp[p] = dtp[p] = 0.0 ;
						dsval = dtval = 1.0 ;
					} else {
						dsp[p] = dsval * (3-p) ;
						dtp[p] = dtval * (3-p) ;
						dsval *= s ;
						dtval *= t ;
					}
				}

				vsp = new THREE.Vector4( sp[0], sp[1], sp[2], sp[3] );
				vtp = new THREE.Vector4( tp[0], tp[1], tp[2], tp[3] );
				vdsp = new THREE.Vector4( dsp[0], dsp[1], dsp[2], dsp[3] );
				vdtp = new THREE.Vector4( dtp[0], dtp[1], dtp[2], dtp[3] );

				// do for x,y,z
				for ( i = 0 ; i < 3 ; i++ ) {
					// multiply power vectors times matrix to get value
					tcoord = vsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					vert[i] = tcoord.dot( vtp );

					// get s and t tangent vectors
					tcoord = vdsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					sdir[i] = tcoord.dot( vtp ) ;

					tcoord = vsp.clone();
					tcoord.applyMatrix4( mgm[i] );
					tdir[i] = tcoord.dot( vdtp ) ;
				}

				// find normal
				vsdir = new THREE.Vector3( sdir[0], sdir[1], sdir[2] );
				vtdir = new THREE.Vector3( tdir[0], tdir[1], tdir[2] );
				norm.crossVectors( vtdir, vsdir );
				norm.normalize();

				// rotate on X axis
				// interestingly, normals need to be reversed; I suspect the patch
				// has opposite handedness from the teapot. Also, Y is up for the teacup.
				normals.push( new THREE.Vector3( -norm.x, -norm.y, -norm.z ) );

				// TODO: check texturing
				uvs.push( new THREE.Vector2( 1-t, 1-s ) );

				// three.js uses Y up, the code makes Y up, all is fine.
				// Move teacup to be centered around origin, three.js style.
				vertOut = new THREE.Vector3( trueSize*vert[0], trueSize*(vert[1]-maxHeight2), trueSize*vert[2] );

				this.vertices.push( vertOut );

			}
		}

		// save the faces
		for ( sstep = 0 ; sstep < this.segments ; sstep++ ) {
			for ( tstep = 0 ; tstep < this.segments ; tstep++ ) {
				v1 = surfCount * vertPerRow * vertPerRow + sstep * vertPerRow + tstep;
				v2 = v1 + 1;
				v3 = v2 + vertPerRow;
				v4 = v1 + vertPerRow;

				if ( notDegenerate ( this.vertices[v1], this.vertices[v2], this.vertices[v3] ) ) {
					this.faces.push( new THREE.Face3( v1, v2, v3, [ normals[v1], normals[v2], normals[v3] ] ) );
					this.faceVertexUvs[ 0 ].push( [ uvs[v1], uvs[v2], uvs[v3] ] );
				}
				if ( notDegenerate ( this.vertices[v1], this.vertices[v3], this.vertices[v4] ) ) {
					this.faces.push( new THREE.Face3( v1, v3, v4, [ normals[v1], normals[v3], normals[v4] ] ) );
					this.faceVertexUvs[ 0 ].push( [ uvs[v1], uvs[v3], uvs[v4] ] );
				}
				//faceCount+=2;
			}
		}
		// increment only if a surface was used
		surfCount++;
	}

	this.computeCentroids();
	this.computeFaceNormals();
	this.mergeVertices();

};


THREE.TeacupGeometry.prototype = Object.create( THREE.Geometry.prototype );
